<?php
require_once '../../src/includes/config.php';

// Vérifier l'authentification
if (!isset($_SESSION['gestionnaire_id'])) {
    header('Location: login.php');
    exit;
}

$role = $_SESSION['role'];
$service_id = $_SESSION['service_id'];

// 🔐 Services autorisés pour le filtre
$services_autorises = ($role === 'admin') ? [1, 2] : [$service_id];
$placeholders_services = str_repeat('?,', count($services_autorises) - 1) . '?';

// Traitement de l'export
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['exporter'])) {
    $type_export = $_POST['type_export'] ?? 'general';
    $date_debut = $_POST['date_debut'] ?? date('Y-m-01');
    $date_fin = $_POST['date_fin'] ?? date('Y-m-d');
    
    // Récupérer les données selon le type d'export
    $donnees = genererDonneesExport($type_export, $date_debut, $date_fin, $pdo, $services_autorises);
    
    // Générer l'export CSV
    exporterCSV($donnees, $type_export);
    exit;
}

// Fonction pour générer les données d'export
function genererDonneesExport($type, $debut, $fin, $pdo, $services_autorises) {
    $placeholders = str_repeat('?,', count($services_autorises) - 1) . '?';
    $params = array_merge([$debut, $fin], $services_autorises);
    
    $donnees = [
        'titre' => '',
        'colonnes' => [],
        'lignes' => [],
        'statistiques' => []
    ];
    
    switch($type) {
        case 'gestionnaires':
            $donnees['titre'] = 'Export des Gestionnaires';
            
            $stmt = $pdo->prepare("
                SELECT g.id, g.nom_complet, g.email, g.role, g.actif, g.date_creation,
                       s.nom_service as service
                FROM gestionnaires g
                LEFT JOIN services s ON g.service_id = s.id
                WHERE g.date_creation BETWEEN ? AND ?
                ORDER BY g.date_creation DESC
            ");
            $stmt->execute([$debut, $fin]);
            $donnees['lignes'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $donnees['colonnes'] = ['ID', 'Nom Complet', 'Email', 'Rôle', 'Actif', 'Date Création', 'Service'];
            break;
            
        case 'dossiers':
            $donnees['titre'] = 'Export des Dossiers';
            
            $stmt = $pdo->prepare("
                SELECT d.id, d.numero_dossier, d.titre, d.description, d.mode_depot,
                       d.nom_auteur, d.matricule_auteur, d.service_auteur, d.lieu,
                       td.libelle as type_dossier,
                       ts.libelle as type_situation,
                       s.nom_service as service_cible,
                       st.libelle as statut,
                       d.date_creation, d.date_maj, d.auteur_notifie
                FROM dossiers d
                LEFT JOIN types_dossier td ON d.type_id = td.id
                LEFT JOIN types_situation ts ON d.type_situation_id = ts.id
                LEFT JOIN services s ON d.service_cible_id = s.id
                LEFT JOIN statuts st ON d.statut_id = st.id
                WHERE d.date_creation BETWEEN ? AND ? AND d.service_cible_id IN ($placeholders)
                ORDER BY d.date_creation DESC
            ");
            $stmt->execute($params);
            $donnees['lignes'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $donnees['colonnes'] = ['ID', 'N° Dossier', 'Titre', 'Description', 'Mode Dépôt', 'Nom Auteur', 'Matricule', 'Service Auteur', 'Lieu', 'Type', 'Type Situation', 'Service Cible', 'Statut', 'Date Création', 'Date MAJ', 'Auteur Notifié'];
            break;
            
        case 'services':
            $donnees['titre'] = 'Export des Services';
            
            $stmt = $pdo->prepare("
                SELECT s.id, s.nom_service, s.email_alerte, s.description,
                       COUNT(DISTINCT g.id) as nb_gestionnaires,
                       COUNT(DISTINCT d.id) as nb_dossiers
                FROM services s
                LEFT JOIN gestionnaires g ON s.id = g.service_id
                LEFT JOIN dossiers d ON s.id = d.service_cible_id
                GROUP BY s.id
                ORDER BY s.id
            ");
            $stmt->execute();
            $donnees['lignes'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $donnees['colonnes'] = ['ID', 'Nom Service', 'Email Alerte', 'Description', 'Nb Gestionnaires', 'Nb Dossiers'];
            break;
            
        case 'pieces_jointes':
            $donnees['titre'] = 'Export des Pièces Jointes';
            
            $stmt = $pdo->prepare("
                SELECT pj.id, pj.nom_fichier, pj.taille, pj.date_upload, pj.hash_sha256,
                       d.numero_dossier, d.titre as dossier_titre
                FROM pieces_jointes pj
                LEFT JOIN dossiers d ON pj.dossier_id = d.id
                WHERE pj.date_upload BETWEEN ? AND ? AND d.service_cible_id IN ($placeholders)
                ORDER BY pj.date_upload DESC
            ");
            $stmt->execute($params);
            $donnees['lignes'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $donnees['colonnes'] = ['ID', 'Nom Fichier', 'Taille (octets)', 'Date Upload', 'Hash SHA256', 'N° Dossier', 'Titre Dossier'];
            break;
            
        case 'consentements':
            $donnees['titre'] = 'Export des Consentements';
            
            $stmt = $pdo->prepare("
                SELECT c.id, c.dossier_id, c.ip_utilisateur, c.user_agent, 
                       c.date_consentement, c.accepte_transfert_etranget,
                       d.numero_dossier
                FROM consentements c
                LEFT JOIN dossiers d ON c.dossier_id = d.id
                WHERE c.date_consentement BETWEEN ? AND ? AND d.service_cible_id IN ($placeholders)
                ORDER BY c.date_consentement DESC
            ");
            $stmt->execute($params);
            $donnees['lignes'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $donnees['colonnes'] = ['ID', 'Dossier ID', 'N° Dossier', 'IP Utilisateur', 'User Agent', 'Date Consentement', 'Accepte Transfert Étranger'];
            break;
            
        case 'types_situation':
            $donnees['titre'] = 'Export des Types de Situation';
            
            $stmt = $pdo->prepare("
                SELECT ts.id, ts.libelle, ts.code, ts.dossier_type_id, ts.actif, ts.created_at, ts.updated_at,
                       td.libelle as type_dossier
                FROM types_situation ts
                LEFT JOIN types_dossier td ON ts.dossier_type_id = td.id
                ORDER BY ts.dossier_type_id, ts.libelle
            ");
            $stmt->execute();
            $donnees['lignes'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $donnees['colonnes'] = ['ID', 'Libellé', 'Code', 'Type Dossier ID', 'Type Dossier', 'Actif', 'Créé le', 'Mis à jour le'];
            break;
            
        default:
            $donnees['titre'] = 'Export Général CI-ENERGIES';
            $donnees['colonnes'] = ['Table', 'Total', 'Dernière MAJ'];
            
            // Statistiques générales
            $tables_info = [
                'gestionnaires' => 'date_creation',
                'dossiers' => 'date_creation', 
                'services' => null,
                'pieces_jointes' => 'date_upload',
                'consentements' => 'date_consentement',
                'types_situation' => 'created_at'
            ];
            
            $donnees['lignes'] = [];
            
            foreach ($tables_info as $table => $colonne_date) {
                // Compter le total
                $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM $table");
                $stmt->execute();
                $total = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Récupérer la date de la dernière entrée si la colonne existe
                $derniere_maj = 'N/A';
                if ($colonne_date) {
                    $stmt = $pdo->prepare("SELECT MAX($colonne_date) as derniere_maj FROM $table");
                    $stmt->execute();
                    $result = $stmt->fetch(PDO::FETCH_ASSOC);
                    $derniere_maj = $result['derniere_maj'] ? date('d/m/Y H:i', strtotime($result['derniere_maj'])) : 'N/A';
                }
                
                $donnees['lignes'][] = [
                    'Table' => ucfirst(str_replace('_', ' ', $table)),
                    'Total' => $total['total'],
                    'Dernière MAJ' => $derniere_maj
                ];
            }
            
            // Ajouter des métriques supplémentaires
            $donnees['lignes'][] = ['Table' => 'Export généré le', 'Total' => '', 'Dernière MAJ' => date('d/m/Y H:i')];
            $donnees['lignes'][] = ['Table' => 'Généré par', 'Total' => '', 'Dernière MAJ' => $_SESSION['nom_complet']];
            break;
    }
    
    return $donnees;
}

// Fonction pour exporter en CSV
function exporterCSV($donnees, $type_export) {
    $filename = 'export_' . $type_export . '_' . date('Y-m-d_H-i') . '.csv';
    
    // Headers pour le téléchargement CSV
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = fopen('php://output', 'w');
    
    // Ajouter l'en-tête UTF-8 BOM pour Excel
    fputs($output, "\xEF\xBB\xBF");
    
    // En-tête informatif
    fputcsv($output, ['CI-ENERGIES - Système d\'Intégrité'], ';');
    fputcsv($output, [$donnees['titre']], ';');
    fputcsv($output, ['Généré le: ' . date('d/m/Y à H:i')], ';');
    fputcsv($output, ['Par: ' . $_SESSION['nom_complet']], ';');
    fputcsv($output, [''], ';'); // Ligne vide
    
    // Écrire les en-têtes de colonnes
    fputcsv($output, $donnees['colonnes'], ';');
    
    // Écrire les données
    foreach ($donnees['lignes'] as $ligne) {
        $row = [];
        foreach ($ligne as $cellule) {
            // Formater les valeurs booléennes
            if ($cellule === true || $cellule === 1) $cellule = 'Oui';
            if ($cellule === false || $cellule === 0) $cellule = 'Non';
            
            // Nettoyer les caractères problématiques pour CSV
            $cellule = str_replace(["\r", "\n", "\t"], ' ', $cellule);
            $cellule = html_entity_decode($cellule, ENT_QUOTES, 'UTF-8');
            
            $row[] = $cellule;
        }
        fputcsv($output, $row, ';');
    }
    
    // Pied de page
    fputcsv($output, [''], ';');
    fputcsv($output, ['© ' . date('Y') . ' CI-ENERGIES - Tous droits réservés'], ';');
    fputcsv($output, ['Document généré automatiquement'], ';');
    
    fclose($output);
    exit;
}
?>

<!-- Le reste du code HTML reste identique, seul le select pour le type d'export doit être mis à jour -->
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Export de Données | CI-ENERGIES</title>
    
    <!-- Bootstrap 5.3.3 -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Icônes Bootstrap -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
    
    <style>
        .export-card {
            background: white;
            border-radius: 15px;
            box-shadow: 0 5px 25px rgba(0,0,0,0.1);
            border: none;
        }
        .form-label {
            font-weight: 600;
            color: #2E7D32;
        }
        .btn-export {
            background: linear-gradient(135deg, #2E7D32, #4CAF50);
            border: none;
            padding: 12px 30px;
            font-weight: 600;
        }
        .preview-box {
            background: #f8f9fa;
            border: 2px dashed #dee2e6;
            border-radius: 10px;
            padding: 2rem;
            text-align: center;
        }
        .format-badge {
            font-size: 0.7rem;
            padding: 0.3em 0.6em;
        }
        .csv-features {
            background: linear-gradient(135deg, #e3f2fd, #f3e5f5);
            border-left: 4px solid #2196f3;
        }
    </style>
</head>
<body>
    <?php include 'admin_header.php'; ?>
    
    <div class="container-fluid p-4">
        <!-- En-tête de page -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h1 class="h3 mb-1">
                    <i class="bi bi-download text-primary me-2"></i>
                    Export de Données
                </h1>
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0">
                        <li class="breadcrumb-item"><a href="dashboard.php">Tableau de bord</a></li>
                        <li class="breadcrumb-item active">Export CSV</li>
                    </ol>
                </nav>
            </div>
            <div>
                <span class="badge bg-success">CSV</span>
            </div>
        </div>

        <div class="alert alert-info mb-4">
            <div class="d-flex align-items-center">
                <div class="flex-shrink-0">
                    <i class="bi bi-info-circle-fill fs-5"></i>
                </div>
                <div class="flex-grow-1 ms-3">
                    <h6 class="alert-heading mb-1">Export CSV Universel</h6>
                    <p class="mb-0">L'export utilise le format CSV, compatible avec Excel, Google Sheets, LibreOffice et tous les tableurs.</p>
                </div>
            </div>
        </div>

        <div class="row g-4">
            <div class="col-lg-8">
                <div class="card border-0 shadow-sm">
                    <div class="card-header bg-white border-bottom">
                        <h5 class="card-title mb-0">
                            <i class="bi bi-gear me-2"></i>
                            Paramètres de l'Export
                        </h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" id="exportForm">
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <label for="type_export" class="form-label fw-semibold">Type d'export</label>
                                    <select class="form-select" id="type_export" name="type_export" required>
                                        <option value="general">Export Général</option>
                                        <option value="gestionnaires">Export Gestionnaires</option>
                                        <option value="dossiers">Export Dossiers</option>
                                        <option value="services">Export Services</option>
                                        <option value="pieces_jointes">Export Pièces Jointes</option>
                                        <option value="consentements">Export Consentements</option>
                                        <option value="types_situation">Export Types de Situation</option>
                                    </select>
                                </div>
                                
                                <div class="col-md-3">
                                    <label for="date_debut" class="form-label fw-semibold">Date de début</label>
                                    <input type="date" class="form-control" id="date_debut" name="date_debut" 
                                           value="<?= date('Y-m-01') ?>" required>
                                </div>
                                
                                <div class="col-md-3">
                                    <label for="date_fin" class="form-label fw-semibold">Date de fin</label>
                                    <input type="date" class="form-control" id="date_fin" name="date_fin" 
                                           value="<?= date('Y-m-d') ?>" required>
                                </div>
                            </div>
                            
                            <div class="mt-4 pt-3 border-top">
                                <button type="submit" name="exporter" class="btn btn-primary">
                                    <i class="bi bi-download me-2"></i>
                                    Exporter en CSV
                                </button>
                                <a href="dashboard.php" class="btn btn-secondary">Annuler</a>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-4">
                <div class="card border-0 shadow-sm">
                    <div class="card-header bg-white border-bottom">
                        <h5 class="card-title mb-0">
                            <i class="bi bi-info-circle me-2"></i>
                            Informations
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="preview-box mb-4">
                            <i class="bi bi-file-earmark-spreadsheet display-4 text-primary mb-3"></i>
                            <h6>Export CSV Optimisé</h6>
                            <p class="text-muted small">
                                Format universel compatible avec Excel, Google Sheets, etc.
                            </p>
                        </div>
                        
                        <div class="alert alert-info">
                            <h6 class="alert-heading">Types d'export disponibles</h6>
                            <ul class="small mb-0">
                                <li><strong>Général</strong> : Statistiques globales du système</li>
                                <li><strong>Gestionnaires</strong> : Liste des utilisateurs administrateurs</li>
                                <li><strong>Dossiers</strong> : Alertes et signalements reçus</li>
                                <li><strong>Services</strong> : Unités organisationnelles</li>
                                <li><strong>Pièces Jointes</strong> : Fichiers attachés aux dossiers</li>
                                <li><strong>Consentements</strong> : Traces des consentements RGPD</li>
                                <li><strong>Types de Situation</strong> : Catégories de signalements</li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <?php include 'admin_footer.php'; ?>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        // Validation des dates
        document.getElementById('exportForm').addEventListener('submit', function(e) {
            const dateDebut = new Date(document.getElementById('date_debut').value);
            const dateFin = new Date(document.getElementById('date_fin').value);
            
            if (dateDebut > dateFin) {
                e.preventDefault();
                alert('La date de début ne peut pas être après la date de fin.');
                return false;
            }
        });
        
        // Mettre à jour la date de fin max à aujourd'hui
        document.getElementById('date_fin').max = new Date().toISOString().split('T')[0];
    </script>
</body>
</html>