
<?php
// save_suggestion.php — version avec gestion du consentement
error_reporting(E_ALL);
ini_set('display_errors', 1);

// === CONNEXION PDO ===
require_once __DIR__ . '/../src/includes/config.php';

if (!isset($pdo) || !($pdo instanceof PDO)) {
    session_start();
    $_SESSION['form_errors'] = ["Erreur de connexion à la base de données."];
    header("Location: suggestion.php");
    exit;
}

// === Récupération des données du formulaire ===
$type_situation_id = filter_input(INPUT_POST, 'type_situation_id', FILTER_VALIDATE_INT);
$description = trim((string)($_POST['description'] ?? ''));
$mode_depot = $_POST['mode_depot'] ?? 'anonyme';

// Validation du consentement
$consent_finalite = isset($_POST['consent_finalite']);
$consent_donnees = isset($_POST['consent_donnees']);
$consent_duree = isset($_POST['consent_duree']);
$consent_retour = isset($_POST['consent_retour']);

// Générer un numéro de dossier
$numero_dossier = (string)($_POST['numero_dossier'] ?? '');
if ($numero_dossier === '') {
    $numero_dossier = genererNumeroDossier('suggestion');
}

// === Validation des données ===
$errors = [];
if (!$type_situation_id) {
    $errors[] = "Le thème de la suggestion est obligatoire.";
}
if (empty($description)) {
    $errors[] = "La description de la suggestion est obligatoire.";
}
if (!$consent_finalite || !$consent_donnees || !$consent_duree || !$consent_retour) {
    $errors[] = "Vous devez accepter tous les points du formulaire de consentement.";
}

if (!empty($errors)) {
    session_start();
    $_SESSION['form_errors'] = $errors;
    $_SESSION['form_data'] = $_POST;
    header("Location: suggestion.php");
    exit;
}

// === Préparation des données ===
// Type de dossier FIXE = 2 (Suggestion)
$type_dossier_id = 2;

// Récupérer le libellé du type de situation
$type_situation_label = "Suggestion";
try {
    $stmtCheck = $pdo->prepare("SELECT libelle FROM types_situation WHERE id = ? AND dossier_type_id = 2 AND actif = 1");
    $stmtCheck->execute([$type_situation_id]);
    $type_situation = $stmtCheck->fetch(PDO::FETCH_ASSOC);
    
    if ($type_situation) {
        $type_situation_label = $type_situation['libelle'];
    }
} catch (Exception $e) {
    error_log("Erreur récupération type situation: " . $e->getMessage());
}

// Générer le titre automatiquement
$titre = "Suggestion - " . $type_situation_label;

// Champs identifiants
$nom_auteur = null;
$matricule_auteur = null;
$service_auteur = null;

if ($mode_depot === 'confidentiel') {
    $nom_auteur = trim((string)($_POST['nom_auteur'] ?? ''));
    $matricule_auteur = trim((string)($_POST['matricule_auteur'] ?? ''));
    $service_auteur = trim((string)($_POST['service_auteur'] ?? ''));
}

$service_cible_id = 2; // Service du Courrier
$statut_id = 1; // Statut "Reçu"

// === Gestion de l'upload de fichier ===
$hasFile = isset($_FILES['fichier']) && is_array($_FILES['fichier']) && $_FILES['fichier']['error'] !== UPLOAD_ERR_NO_FILE;
$uploadOk = false;
$uploadRel = null;
$hash = null;
$origName = null;
$sizeBytes = null;

if ($hasFile) {
    $f = $_FILES['fichier'];
    
    if ($f['error'] !== UPLOAD_ERR_OK) {
        session_start();
        $_SESSION['form_errors'] = ["Erreur lors de l'upload du fichier (code {$f['error']})."];
        $_SESSION['form_data'] = $_POST;
        header("Location: suggestion.php");
        exit;
    }

    $origName = $f['name'];
    $sizeBytes = (int)$f['size'];
    $tmp = $f['tmp_name'];

    // Vérification taille (10MB max)
    if ($sizeBytes > 10 * 1024 * 1024) {
        session_start();
        $_SESSION['form_errors'] = ["Le fichier est trop volumineux (> 10 Mo)."];
        $_SESSION['form_data'] = $_POST;
        header("Location: suggestion.php");
        exit;
    }

    // Vérification extension
    $ext = strtolower(pathinfo($origName, PATHINFO_EXTENSION));
    $allowed = ['pdf', 'jpg', 'jpeg', 'png', 'doc', 'docx'];
    
    if ($ext && !in_array($ext, $allowed, true)) {
        session_start();
        $_SESSION['form_errors'] = ["Type de fichier non autorisé (.$ext). Formats acceptés: " . implode(', ', $allowed)];
        $_SESSION['form_data'] = $_POST;
        header("Location: suggestion.php");
        exit;
    }

    // Préparation répertoire d'upload
    $baseUploadDir = __DIR__ . '/uploads';
    $subDir = date('Y') . '/' . date('m');
    $targetDir = $baseUploadDir . '/' . $subDir;
    
    if (!is_dir($targetDir) && !mkdir($targetDir, 0775, true) && !is_dir($targetDir)) {
        session_start();
        $_SESSION['form_errors'] = ["Impossible de créer le répertoire de stockage."];
        $_SESSION['form_data'] = $_POST;
        header("Location: suggestion.php");
        exit;
    }

    // Génération nom sécurisé
    $safe = preg_replace('/[^a-zA-Z0-9_\-]/', '_', pathinfo($origName, PATHINFO_FILENAME)) ?: 'pj';
    $serverName = $safe . '-' . bin2hex(random_bytes(4)) . ($ext ? ".$ext" : '');
    $targetPath = $targetDir . '/' . $serverName;
    $uploadRel = 'uploads/' . $subDir . '/' . $serverName;

    // Hash et déplacement
    $hash = hash_file('sha256', $tmp);
    
    if (!move_uploaded_file($tmp, $targetPath)) {
        session_start();
        $_SESSION['form_errors'] = ["Erreur lors de l'enregistrement du fichier."];
        $_SESSION['form_data'] = $_POST;
        header("Location: suggestion.php");
        exit;
    }
    
    $uploadOk = true;
}

// === Insertion en base ===
try {
    $pdo->beginTransaction();

    // Vérifier si la colonne type_situation_id existe
    $hasTypeSituationColumn = false;
    try {
        $checkColumn = $pdo->query("SHOW COLUMNS FROM dossiers LIKE 'type_situation_id'");
        $hasTypeSituationColumn = $checkColumn->rowCount() > 0;
    } catch (Exception $e) {
        $hasTypeSituationColumn = false;
    }

    // Construction dynamique de la requête SQL
    $columns = [
        'numero_dossier', 
        'type_id', 
        'titre', 
        'description', 
        'mode_depot',
        'service_cible_id', 
        'statut_id', 
        'date_creation'
    ];
    
    $placeholders = [
        ':numero', 
        ':type_dossier', 
        ':titre', 
        ':descr', 
        ':mode',
        ':service', 
        ':statut', 
        'NOW()'
    ];
    
    $params = [
        ':numero'       => $numero_dossier,
        ':type_dossier' => $type_dossier_id,
        ':titre'        => $titre,
        ':descr'        => $description,
        ':mode'         => $mode_depot,
        ':service'      => $service_cible_id,
        ':statut'       => $statut_id,
    ];

    // Ajouter type_situation_id si la colonne existe
    if ($hasTypeSituationColumn) {
        $columns[] = 'type_situation_id';
        $placeholders[] = ':type_situation';
        $params[':type_situation'] = $type_situation_id;
    }

    // Ajouter les champs d'identité si mode confidentiel
    if ($mode_depot === 'confidentiel') {
        $columns[] = 'nom_auteur';
        $placeholders[] = ':nom_auteur';
        $params[':nom_auteur'] = $nom_auteur;

        $columns[] = 'matricule_auteur';
        $placeholders[] = ':matricule_auteur';
        $params[':matricule_auteur'] = $matricule_auteur;

        $columns[] = 'service_auteur';
        $placeholders[] = ':service_auteur';
        $params[':service_auteur'] = $service_auteur;
    }

    // Construction de la requête finale
    $sqlDossier = "
        INSERT INTO dossiers (" . implode(', ', $columns) . ")
        VALUES (" . implode(', ', $placeholders) . ")
    ";
    
    $stmt = $pdo->prepare($sqlDossier);
    $stmt->execute($params);
    
    $dossier_id = (int)$pdo->lastInsertId();

    // Insertion pièce jointe si présente
    if ($uploadOk) {
        // Vérifier la structure de la table pieces_jointes
        $checkPiecesColumns = $pdo->query("SHOW COLUMNS FROM pieces_jointes");
        $piecesColumns = $checkPiecesColumns->fetchAll(PDO::FETCH_COLUMN);
        
        // Construction dynamique pour pieces_jointes
        $pjColumns = ['dossier_id', 'nom_fichier', 'chemin_stockage'];
        $pjPlaceholders = [':dossier_id', ':nom_fichier', ':chemin'];
        $pjParams = [
            ':dossier_id'  => $dossier_id,
            ':nom_fichier' => $origName,
            ':chemin'      => $uploadRel,
        ];

        // Ajouter taille si la colonne existe
        if (in_array('taille', $piecesColumns)) {
            $pjColumns[] = 'taille';
            $pjPlaceholders[] = ':taille';
            $pjParams[':taille'] = $sizeBytes;
        }
        
        // Ajouter hash_sha256 si la colonne existe
        if (in_array('hash_sha256', $piecesColumns)) {
            $pjColumns[] = 'hash_sha256';
            $pjPlaceholders[] = ':hash';
            $pjParams[':hash'] = $hash;
        }

        $sqlPiece = "
            INSERT INTO pieces_jointes (" . implode(', ', $pjColumns) . ")
            VALUES (" . implode(', ', $pjPlaceholders) . ")
        ";
        
        $stmt2 = $pdo->prepare($sqlPiece);
        $stmt2->execute($pjParams);
    }

    // Enregistrement du consentement
    $ip_utilisateur = $_SERVER['REMOTE_ADDR'] ?? null;
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? null;

    $sqlConsentement = "
        INSERT INTO consentements (dossier_id, ip_utilisateur, user_agent, accepte_transfert_etranget)
        VALUES (?, ?, ?, 0)
    ";
    $stmtConsentement = $pdo->prepare($sqlConsentement);
    $stmtConsentement->execute([$dossier_id, $ip_utilisateur, $user_agent]);

    $pdo->commit();

    // Redirection vers la page de succès
    header("Location: merci.php?numero=" . urlencode($numero_dossier) . "&type=suggestion");
    exit;

} catch (Throwable $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    // Rediriger vers le formulaire avec erreur
    session_start();
    $_SESSION['form_errors'] = ["Erreur lors de l'enregistrement : " . $e->getMessage()];
    $_SESSION['form_data'] = $_POST;
    header("Location: suggestion.php");
    exit;
}
?>
