<?php
// utilisateurs.php
require_once '../../src/includes/config.php';

// Vérifier l'authentification et les permissions
if (!isset($_SESSION['gestionnaire_id'])) {
    header('Location: login.php');
    exit;
}

// Vérifier si l'utilisateur est admin
$isAdmin = ($_SESSION['role'] === 'admin');

if (!$isAdmin) {
    header('Location: dashboard.php');
    exit;
}

// Variables pour les messages
$success_message = '';
$error_message = '';

// Traitement de l'ajout d'un utilisateur
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajouter_utilisateur'])) {
    $nom_complet = trim($_POST['nom_complet']);
    $email = trim($_POST['email']);
    $mot_de_passe = $_POST['mot_de_passe'];
    $service_id = intval($_POST['service_id']);
    $role = $_POST['role'];
    
    // Validation
    if (empty($nom_complet) || empty($email) || empty($mot_de_passe) || empty($service_id)) {
        $error_message = "Tous les champs obligatoires doivent être remplis.";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error_message = "L'adresse email n'est pas valide.";
    } elseif (strlen($mot_de_passe) < 6) {
        $error_message = "Le mot de passe doit contenir au moins 6 caractères.";
    } else {
        try {
            // Vérifier si l'email existe déjà
            $stmt = $pdo->prepare("SELECT id FROM gestionnaires WHERE email = ?");
            $stmt->execute([$email]);
            if ($stmt->fetch()) {
                $error_message = "Un compte avec cet email existe déjà.";
            } else {
                // Hasher le mot de passe
                $mot_de_passe_hash = password_hash($mot_de_passe, PASSWORD_DEFAULT);
                
                // Insérer le nouvel utilisateur
                $stmt = $pdo->prepare("
                    INSERT INTO gestionnaires (nom_complet, email, mot_de_passe, service_id, role, actif, date_creation)
                    VALUES (?, ?, ?, ?, ?, 1, NOW())
                ");
                $stmt->execute([$nom_complet, $email, $mot_de_passe_hash, $service_id, $role]);
                
                $success_message = "Utilisateur créé avec succès !";
            }
        } catch (Exception $e) {
            $error_message = "Erreur lors de la création de l'utilisateur : " . $e->getMessage();
        }
    }
}

// Traitement de la modification d'un utilisateur
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['modifier_utilisateur'])) {
    $user_id = intval($_POST['user_id']);
    $nom_complet = trim($_POST['nom_complet']);
    $email = trim($_POST['email']);
    $service_id = intval($_POST['service_id']);
    $role = $_POST['role'];
    $actif = isset($_POST['actif']) ? 1 : 0;
    
    // Validation
    if (empty($nom_complet) || empty($email) || empty($service_id)) {
        $error_message = "Tous les champs obligatoires doivent être remplis.";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error_message = "L'adresse email n'est pas valide.";
    } else {
        try {
            // Vérifier si l'email existe déjà pour un autre utilisateur
            $stmt = $pdo->prepare("SELECT id FROM gestionnaires WHERE email = ? AND id != ?");
            $stmt->execute([$email, $user_id]);
            if ($stmt->fetch()) {
                $error_message = "Un autre compte utilise déjà cet email.";
            } else {
                // Mettre à jour l'utilisateur
                $stmt = $pdo->prepare("
                    UPDATE gestionnaires 
                    SET nom_complet = ?, email = ?, service_id = ?, role = ?, actif = ?
                    WHERE id = ?
                ");
                $stmt->execute([$nom_complet, $email, $service_id, $role, $actif, $user_id]);
                
                $success_message = "Utilisateur modifié avec succès !";
            }
        } catch (Exception $e) {
            $error_message = "Erreur lors de la modification de l'utilisateur : " . $e->getMessage();
        }
    }
}

// Traitement de la réinitialisation du mot de passe
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['reinitialiser_mdp'])) {
    $user_id = intval($_POST['user_id']);
    $nouveau_mot_de_passe = $_POST['nouveau_mot_de_passe'];
    
    if (strlen($nouveau_mot_de_passe) < 6) {
        $error_message = "Le mot de passe doit contenir au moins 6 caractères.";
    } else {
        try {
            $mot_de_passe_hash = password_hash($nouveau_mot_de_passe, PASSWORD_DEFAULT);
            $stmt = $pdo->prepare("UPDATE gestionnaires SET mot_de_passe = ? WHERE id = ?");
            $stmt->execute([$mot_de_passe_hash, $user_id]);
            
            $success_message = "Mot de passe réinitialisé avec succès !";
        } catch (Exception $e) {
            $error_message = "Erreur lors de la réinitialisation du mot de passe : " . $e->getMessage();
        }
    }
}

// Récupérer la liste des gestionnaires
try {
    $stmt = $pdo->prepare("
        SELECT g.*, s.nom_service 
        FROM gestionnaires g 
        LEFT JOIN services s ON g.service_id = s.id 
        ORDER BY g.date_creation DESC
    ");
    $stmt->execute();
    $gestionnaires = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $error_message = "Erreur lors du chargement des utilisateurs : " . $e->getMessage();
    $gestionnaires = [];
}

// Récupérer la liste des services
try {
    $stmt = $pdo->prepare("SELECT id, nom_service FROM services ORDER BY nom_service");
    $stmt->execute();
    $services = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $services = [];
}

$base = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/');
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Gestion des Utilisateurs | CI-ENERGIES</title>
    
    <!-- Bootstrap 5.3.3 -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Icônes Bootstrap -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
    
    <style>
        .users-card {
            background: white;
            border-radius: 15px;
            box-shadow: 0 5px 25px rgba(0,0,0,0.1);
            border: none;
        }
        .btn-primary-custom {
            background: linear-gradient(135deg, #2E7D32, #4CAF50);
            border: none;
            padding: 10px 20px;
            font-weight: 600;
        }
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: linear-gradient(135deg, #2E7D32, #4CAF50);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
            font-size: 14px;
        }
        .status-badge {
            font-size: 0.75rem;
            padding: 0.35em 0.65em;
        }
        .action-buttons .btn {
            padding: 0.25rem 0.5rem;
            font-size: 0.875rem;
        }
        .table th {
            background: #f8f9fa;
            font-weight: 600;
            color: #2E7D32;
        }
        .search-box {
            max-width: 300px;
        }
    </style>
</head>
<body>
    <?php 
    // Inclure le header si le fichier existe
    $headerPath = '../includes/header.php';
    if (file_exists($headerPath)) {
        include $headerPath;
    }
    ?>
    
    <div class="app-wrapper">
        <main class="container-fluid py-4">
            <!-- En-tête de page -->
            <div class="page-header bg-white rounded-3 shadow-sm mb-4">
                <div class="container-fluid py-3">
                    <div class="row align-items-center">
                        <div class="col">
                            <h1 class="page-title h3 mb-0">
                                <i class="bi bi-people text-primary me-2"></i>
                                Gestion des Utilisateurs
                            </h1>
                            <nav aria-label="breadcrumb">
                                <ol class="breadcrumb mb-0">
                                    <li class="breadcrumb-item"><a href="dashboard.php">Tableau de bord</a></li>
                                    <li class="breadcrumb-item active">Utilisateurs</li>
                                </ol>
                            </nav>
                        </div>
                        <div class="col-auto">
                            <button class="btn btn-primary-custom text-white" data-bs-toggle="modal" data-bs-target="#addUserModal">
                                <i class="bi bi-person-plus me-2"></i>
                                Nouvel Utilisateur
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Messages d'alerte -->
            <?php if ($success_message): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bi bi-check-circle-fill me-2"></i>
                <?= htmlspecialchars($success_message) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php endif; ?>

            <?php if ($error_message): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="bi bi-exclamation-triangle-fill me-2"></i>
                <?= htmlspecialchars($error_message) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php endif; ?>

            <div class="row">
                <div class="col-12">
                    <div class="card users-card">
                        <div class="card-header bg-white py-3">
                            <div class="row align-items-center">
                                <div class="col">
                                    <h5 class="card-title mb-0">
                                        <i class="bi bi-list-ul me-2"></i>
                                        Liste des Utilisateurs
                                    </h5>
                                </div>
                                <div class="col-auto">
                                    <div class="input-group search-box">
                                        <span class="input-group-text"><i class="bi bi-search"></i></span>
                                        <input type="text" class="form-control" placeholder="Rechercher..." id="searchInput">
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-hover" id="usersTable">
                                    <thead>
                                        <tr>
                                            <th>Utilisateur</th>
                                            <th>Email</th>
                                            <th>Service</th>
                                            <th>Rôle</th>
                                            <th>Statut</th>
                                            <th>Date Création</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($gestionnaires as $user): ?>
                                        <tr>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="user-avatar me-3">
                                                        <?= strtoupper(substr($user['nom_complet'], 0, 2)) ?>
                                                    </div>
                                                    <div>
                                                        <div class="fw-semibold"><?= htmlspecialchars($user['nom_complet']) ?></div>
                                                        <small class="text-muted">ID: <?= $user['id'] ?></small>
                                                    </div>
                                                </div>
                                            </td>
                                            <td><?= htmlspecialchars($user['email']) ?></td>
                                            <td><?= htmlspecialchars($user['nom_service'] ?? 'Non assigné') ?></td>
                                            <td>
                                                <span class="badge <?= $user['role'] === 'admin' ? 'bg-warning' : 'bg-secondary' ?> status-badge">
                                                    <?= $user['role'] === 'admin' ? 'Administrateur' : 'Gestionnaire' ?>
                                                </span>
                                            </td>
                                            <td>
                                                <span class="badge <?= $user['actif'] ? 'bg-success' : 'bg-danger' ?> status-badge">
                                                    <?= $user['actif'] ? 'Actif' : 'Inactif' ?>
                                                </span>
                                            </td>
                                            <td>
                                                <small class="text-muted">
                                                    <?= date('d/m/Y', strtotime($user['date_creation'])) ?>
                                                </small>
                                            </td>
                                            <td>
                                                <div class="action-buttons">
                                                    <button class="btn btn-sm btn-outline-primary" 
                                                            data-bs-toggle="modal" 
                                                            data-bs-target="#editUserModal"
                                                            data-user-id="<?= $user['id'] ?>"
                                                            data-nom-complet="<?= htmlspecialchars($user['nom_complet']) ?>"
                                                            data-email="<?= htmlspecialchars($user['email']) ?>"
                                                            data-service-id="<?= $user['service_id'] ?>"
                                                            data-role="<?= $user['role'] ?>"
                                                            data-actif="<?= $user['actif'] ?>"
                                                            onclick="fillEditForm(this)">
                                                        <i class="bi bi-pencil"></i>
                                                    </button>
                                                    <button class="btn btn-sm btn-outline-warning" 
                                                            data-bs-toggle="modal" 
                                                            data-bs-target="#resetPasswordModal"
                                                            data-user-id="<?= $user['id'] ?>"
                                                            data-nom-complet="<?= htmlspecialchars($user['nom_complet']) ?>"
                                                            onclick="fillResetForm(this)">
                                                        <i class="bi bi-key"></i>
                                                    </button>
                                                    <?php if ($user['id'] != $_SESSION['gestionnaire_id']): ?>
                                                    <button class="btn btn-sm btn-outline-danger" 
                                                            onclick="confirmDelete(<?= $user['id'] ?>, '<?= htmlspecialchars($user['nom_complet']) ?>')">
                                                        <i class="bi bi-trash"></i>
                                                    </button>
                                                    <?php else: ?>
                                                    <span class="badge bg-light text-muted">Vous</span>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                        
                                        <?php if (empty($gestionnaires)): ?>
                                        <tr>
                                            <td colspan="7" class="text-center py-4 text-muted">
                                                <i class="bi bi-people display-4 d-block mb-2"></i>
                                                Aucun utilisateur trouvé
                                            </td>
                                        </tr>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <!-- Statistiques -->
                            <div class="row mt-4">
                                <div class="col-md-3">
                                    <div class="card bg-light">
                                        <div class="card-body text-center">
                                            <h3 class="text-primary mb-0"><?= count($gestionnaires) ?></h3>
                                            <small class="text-muted">Total Utilisateurs</small>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="card bg-light">
                                        <div class="card-body text-center">
                                            <h3 class="text-success mb-0">
                                                <?= count(array_filter($gestionnaires, fn($u) => $u['actif'])) ?>
                                            </h3>
                                            <small class="text-muted">Utilisateurs Actifs</small>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="card bg-light">
                                        <div class="card-body text-center">
                                            <h3 class="text-warning mb-0">
                                                <?= count(array_filter($gestionnaires, fn($u) => $u['role'] === 'admin')) ?>
                                            </h3>
                                            <small class="text-muted">Administrateurs</small>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="card bg-light">
                                        <div class="card-body text-center">
                                            <h3 class="text-info mb-0"><?= count($services) ?></h3>
                                            <small class="text-muted">Services</small>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Modal Ajout Utilisateur -->
    <div class="modal fade" id="addUserModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <div class="modal-header">
                        <h5 class="modal-title">
                            <i class="bi bi-person-plus me-2"></i>
                            Nouvel Utilisateur
                        </h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="mb-3">
                            <label for="nom_complet" class="form-label">Nom Complet *</label>
                            <input type="text" class="form-control" id="nom_complet" name="nom_complet" required>
                        </div>
                        <div class="mb-3">
                            <label for="email" class="form-label">Email *</label>
                            <input type="email" class="form-control" id="email" name="email" required>
                        </div>
                        <div class="mb-3">
                            <label for="mot_de_passe" class="form-label">Mot de passe *</label>
                            <input type="password" class="form-control" id="mot_de_passe" name="mot_de_passe" required minlength="6">
                            <div class="form-text">Minimum 6 caractères</div>
                        </div>
                        <div class="mb-3">
                            <label for="service_id" class="form-label">Service *</label>
                            <select class="form-select" id="service_id" name="service_id" required>
                                <option value="">Sélectionner un service</option>
                                <?php foreach ($services as $service): ?>
                                <option value="<?= $service['id'] ?>"><?= htmlspecialchars($service['nom_service']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label for="role" class="form-label">Rôle *</label>
                            <select class="form-select" id="role" name="role" required>
                                <option value="gestionnaire">Gestionnaire</option>
                                <option value="admin">Administrateur</option>
                            </select>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Annuler</button>
                        <button type="submit" name="ajouter_utilisateur" class="btn btn-primary-custom text-white">Créer l'utilisateur</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Modal Modification Utilisateur -->
    <div class="modal fade" id="editUserModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <input type="hidden" name="user_id" id="edit_user_id">
                    <div class="modal-header">
                        <h5 class="modal-title">
                            <i class="bi bi-pencil me-2"></i>
                            Modifier l'Utilisateur
                        </h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="mb-3">
                            <label for="edit_nom_complet" class="form-label">Nom Complet *</label>
                            <input type="text" class="form-control" id="edit_nom_complet" name="nom_complet" required>
                        </div>
                        <div class="mb-3">
                            <label for="edit_email" class="form-label">Email *</label>
                            <input type="email" class="form-control" id="edit_email" name="email" required>
                        </div>
                        <div class="mb-3">
                            <label for="edit_service_id" class="form-label">Service *</label>
                            <select class="form-select" id="edit_service_id" name="service_id" required>
                                <option value="">Sélectionner un service</option>
                                <?php foreach ($services as $service): ?>
                                <option value="<?= $service['id'] ?>"><?= htmlspecialchars($service['nom_service']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label for="edit_role" class="form-label">Rôle *</label>
                            <select class="form-select" id="edit_role" name="role" required>
                                <option value="gestionnaire">Gestionnaire</option>
                                <option value="admin">Administrateur</option>
                            </select>
                        </div>
                        <div class="form-check form-switch">
                            <input class="form-check-input" type="checkbox" id="edit_actif" name="actif" value="1">
                            <label class="form-check-label" for="edit_actif">Compte actif</label>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Annuler</button>
                        <button type="submit" name="modifier_utilisateur" class="btn btn-primary-custom text-white">Enregistrer</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Modal Réinitialisation Mot de Passe -->
    <div class="modal fade" id="resetPasswordModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <input type="hidden" name="user_id" id="reset_user_id">
                    <div class="modal-header">
                        <h5 class="modal-title">
                            <i class="bi bi-key me-2"></i>
                            Réinitialiser le Mot de Passe
                        </h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle me-2"></i>
                            Réinitialisation du mot de passe pour : <strong id="reset_user_name"></strong>
                        </div>
                        <div class="mb-3">
                            <label for="nouveau_mot_de_passe" class="form-label">Nouveau mot de passe *</label>
                            <input type="password" class="form-control" id="nouveau_mot_de_passe" name="nouveau_mot_de_passe" required minlength="6">
                            <div class="form-text">Minimum 6 caractères</div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Annuler</button>
                        <button type="submit" name="reinitialiser_mdp" class="btn btn-warning text-white">Réinitialiser</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        // Recherche en temps réel
        document.getElementById('searchInput').addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = document.querySelectorAll('#usersTable tbody tr');
            
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                row.style.display = text.includes(searchTerm) ? '' : 'none';
            });
        });

        // Remplir le formulaire de modification
        function fillEditForm(button) {
            const userId = button.getAttribute('data-user-id');
            const nomComplet = button.getAttribute('data-nom-complet');
            const email = button.getAttribute('data-email');
            const serviceId = button.getAttribute('data-service-id');
            const role = button.getAttribute('data-role');
            const actif = button.getAttribute('data-actif');
            
            document.getElementById('edit_user_id').value = userId;
            document.getElementById('edit_nom_complet').value = nomComplet;
            document.getElementById('edit_email').value = email;
            document.getElementById('edit_service_id').value = serviceId;
            document.getElementById('edit_role').value = role;
            document.getElementById('edit_actif').checked = actif === '1';
        }

        // Remplir le formulaire de réinitialisation
        function fillResetForm(button) {
            const userId = button.getAttribute('data-user-id');
            const nomComplet = button.getAttribute('data-nom-complet');
            
            document.getElementById('reset_user_id').value = userId;
            document.getElementById('reset_user_name').textContent = nomComplet;
            document.getElementById('nouveau_mot_de_passe').value = '';
        }

        // Confirmation de suppression
        function confirmDelete(userId, userName) {
            if (confirm(`Êtes-vous sûr de vouloir supprimer l'utilisateur "${userName}" ?\n\nCette action est irréversible.`)) {
                // Ici vous pourriez ajouter une requête AJAX pour la suppression
                // Pour l'instant, nous n'avons pas implémenté la suppression pour des raisons de sécurité
                alert('Fonction de suppression à implémenter avec sécurité.');
            }
        }

        // Générer un mot de passe aléatoire
        function generatePassword() {
            const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!@#$%^&*';
            let password = '';
            for (let i = 0; i < 12; i++) {
                password += chars.charAt(Math.floor(Math.random() * chars.length));
            }
            document.getElementById('mot_de_passe').value = password;
            document.getElementById('nouveau_mot_de_passe').value = password;
        }
    </script>
</body>
</html>